<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\WaitingQueue;
use App\Models\Tenant;
use App\Models\ActivityLog;
use App\Models\Customer;
use App\Models\DailyStatistic;
use Carbon\Carbon;
use App\Events\QueueUpdated;
use Illuminate\Support\Facades\Auth;
use App\Models\User;

class DashboardController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    public function index()
    {

        $userinfo = User::find(Auth::id());
        $tenant_id= $userinfo->tenant_id;
        $tenant = Tenant::find($tenant_id);
        $uuid= $tenant->uuid;
        // Get current waiting queues
        $appointmentQueue = WaitingQueue::with('customer')
            ->appointments()
            ->waiting()
            ->whereDate('appointment_time', today())
            ->orderBy('appointment_time')
            ->get();

        $walkInQueue = WaitingQueue::with('customer')
            ->walkIns()
            ->waiting()
            ->orderBy('position')
            ->get();

        // Get currently serving customer
        $currentlyServing = WaitingQueue::with('customer')
            ->where('status', 'serving')
            ->first();

        // Get today's statistics
        $today = Carbon::today();
        $todayStats = [
            'total_customers' => WaitingQueue::whereDate('created_at', $today)->count(),
            'completed_customers' => WaitingQueue::whereDate('created_at', $today)
                ->where('status', 'completed')->count(),
            'waiting_customers' => WaitingQueue::where('status', 'waiting')->count(),
            'average_wait_time' => $this->calculateAverageWaitTime()
        ];

        return view('dashboard.index', compact(
            'appointmentQueue',
            'walkInQueue',
            'currentlyServing',
            'todayStats',
            'uuid',
        ));
    }

    public function getNextCustomer(Request $request)
    {
        $nextCustomer = $this->determineNextCustomer();

        // Mark current serving customer as completed if exists
        $currentServing = WaitingQueue::where('status', 'serving')->first();
        if ($currentServing) {
            $currentServing->update([
                'status' => 'completed',
                'completed_at' => now()
            ]);
        }

        if (!$nextCustomer) {
            return response()->json([
                'success' => false,
                'message' => 'لا يوجد عملاء في قائمة الانتظار'
            ]);
        }

        // Mark next customer as serving
        $nextCustomer->update([
            'status' => 'serving',
            'served_by' => auth()->id(),
            'served_at' => now()
        ]);

        // Broadcast WebSocket update AFTER database changes are committed
        broadcast(new QueueUpdated(auth()->user()->tenant_id));

        return response()->json([
            'success' => true,
            'customer' => $nextCustomer->load('customer'),
            'message' => 'تم استدعاء العميل التالي: ' . $nextCustomer->customer->name
        ]);
    }

    private function determineNextCustomer()
    {
        $now = Carbon::now();

        // First, check for due appointments (within 15 minutes of appointment time)
        $dueAppointments = WaitingQueue::with('customer')
        ->appointments()
        ->waiting()
        ->whereNotNull('appointment_time')
        ->whereDate('appointment_time', today()) // إذا كنت تريد فقط مواعيد اليوم
        ->where('appointment_time', '<=', Carbon::now())
        ->orderBy('appointment_time')
        ->get();

if ($dueAppointments->isNotEmpty()) {
    return $dueAppointments->first(); // أو return $dueAppointments;
}

        // If no due appointments, get next walk-in customer
        return WaitingQueue::with('customer')
            ->walkIns()
            ->waiting()
            ->orderBy('position')
            ->first();
    }

    private function calculateAverageWaitTime()
    {
        $today = Carbon::today();
        $completedToday = WaitingQueue::whereDate('created_at', $today)
            ->where('status', 'completed')
            ->whereNotNull('served_at')
            ->whereNotNull('arrival_time')
            ->get();

        if ($completedToday->isEmpty()) {
            return 0;
        }

        $totalWaitTime = $completedToday->sum(function ($queue) {
            return Carbon::parse($queue->served_at)->diffInMinutes(Carbon::parse($queue->arrival_time));
        });

        return round($totalWaitTime / $completedToday->count(), 1);
    }

    public function updateOrder(Request $request)
    {
        $request->validate([
            'id' => 'required|exists:waiting_queue,id',
            'new_order' => 'required|integer|min:1',
        ]);

        $entry = WaitingQueue::findOrFail($request->id);
        $tenantId = $entry->tenant_id;
        $type = $entry->type;

        // استرجاع قائمة الانتظار من نفس النوع والحالة
        $queue = WaitingQueue::where('tenant_id', $tenantId)
            ->where('type', $type)
            ->where('status', 'waiting')
            ->orderBy('position')
            ->get();

        // التحقق من أن الرتبة الجديدة لا تتجاوز عدد العناصر
        $maxOrder = $queue->count();
        if ($request->new_order > $maxOrder) {
            return back()->with('error', "الرتبة الجديدة غير صالحة. يوجد حالياً فقط {$maxOrder} زبائن في الانتظار.");
        }

        // إزالة الزبون من الترتيب الحالي
        $queue = $queue->filter(fn($q) => $q->id !== $entry->id)->values();

        // إدراج الزبون في الترتيب الجديد
        $newIndex = $request->new_order - 1;
        $queue->splice($newIndex, 0, [$entry]);

        ActivityLog::create([
            'user_id' => Auth::id(),
            'action' => 'update_position',
            'description' => "تحديث ترتيب الزبون {$entry->customer->name} إلى {$request->new_order}.",
            'ip_address' => $request->ip(),
            'user_agent' => $request->userAgent(),
        ]);

        // تحديث `position`
        foreach ($queue as $i => $item) {
            $item->position = $i + 1;
            $item->save();
        }

        // Broadcast WebSocket update after position changes
        broadcast(new QueueUpdated(auth()->user()->tenant_id));

        return back()->with('success', 'تم تغيير ترتيب الزبون بنجاح دون تعديل الوقت.');
    }


    public function updateAppointment(Request $request)
{
    $request->validate([
        'id' => 'required',
        'new_time' => 'required|date',
    ], [
        'id.required' => ' مطلوب.',
        'new_time.required' => 'وقت الموعد مطلوب.',
        'new_time.date' => 'وقت الموعد غير صالح.',
    ]);

    $queueEntry = WaitingQueue::where('type', 'appointment')->where('status', 'waiting')->findOrFail($request->id);

    $queueEntry->update([
        'appointment_time' => $request->new_time,
    ]);

    ActivityLog::create([
        'user_id' => Auth::id(),
        'action' => 'update_appointment_time',
        'description' => "تحديث موعد العميل {$queueEntry->customer->name} إلى {$request->new_time}.",
        'ip_address' => $request->ip(),
        'user_agent' => $request->userAgent(),
    ]);

    // Broadcast WebSocket update after appointment time change
    broadcast(new QueueUpdated(auth()->user()->tenant_id));

    return back()->with('success',  "تم تحديث الموعد بنجاح.");
}

    public function delete(Request $request,$id)
    {
        $entry = WaitingQueue::findOrFail($id);

        // احفظ التاريخ قبل الحذف
        $date = $entry->created_at->toDateString();

        // احذف الإدخال
        $entry->delete();
        $type = $entry->type;
        // إعادة ترتيب الزبائن المتبقين بنفس اليوم والحالة "waiting"
        $waitingEntries = WaitingQueue::whereDate('created_at', $date)
            ->where('status', 'waiting')
            ->where('type', $type)
            ->orderBy('position')
            ->get();

        $position = 1;
        foreach ($waitingEntries as $item) {
            $item->update(['position' => $position++]);
        }

        ActivityLog::create([
            'user_id' => Auth::id(),
            'action' => 'Delete_queue',
            'description' => " {$entry->customer->name} وقت:  {$entry->arrival_time}.",
            'ip_address' => $request->ip(),
            'user_agent' => $request->userAgent(),
        ]);

        // Broadcast WebSocket update after queue entry deletion
        broadcast(new QueueUpdated(auth()->user()->tenant_id));

        return response()->json(['status' => 'success']);
    }

    public function delete_appointment(Request $request,$id)
{

    $entry = WaitingQueue::findOrFail($id);

    ActivityLog::create([
        'user_id' => Auth::id(),
        'action' => 'Delete_appointment',
        'description' => " {$entry->customer->name} موعد:  {$entry->appointment_time}.",
        'ip_address' => $request->ip(),
        'user_agent' => $request->userAgent(),
    ]);

    $entry->delete();

    // Broadcast WebSocket update after appointment deletion
    broadcast(new QueueUpdated(auth()->user()->tenant_id));

    return response()->json(['status' => 'success']);
}

public function complete($id)
{
    $queue = WaitingQueue::findOrFail($id);
    $queue->status = 'completed';
    $queue->completed_at = now();
    $queue->save();

    // Broadcast WebSocket update after marking as completed
    broadcast(new QueueUpdated(auth()->user()->tenant_id));

    return redirect()->back()->with('success', __('dashboard.marked_completed'));
}

}
