<?php
namespace App\Http\Controllers;

use App\Models\DisplayDevice;
use App\Models\ShortLink;
use App\Models\WaitingQueue;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class DisplayController extends Controller
{
    protected function resolveDeviceOrAbort(string $device_token): DisplayDevice
    {
        $device = DisplayDevice::where('device_token', $device_token)->first();
        abort_if(!$device, 404);

        // سجل آخر IP (اختياري)
        $device->update(['last_ip' => request()->ip()]);

        return $device;
    }



    public function index(string $device_token)
    {
        $device = $this->resolveDeviceOrAbort($device_token);

        if (!$device->tenant_id || $device->is_authorized == 'non') {
            // جهاز غير مربوط → أعرض شاشة QR
            return redirect()->route('display.pairing', $device->device_token);
        }

        return view('display.index', [
            'device' => $device,
            'tenantId' => $device->tenant_id,
        ]);
    }

    public function data(string $device_token)
    {
        $device = $this->resolveDeviceOrAbort($device_token);
        abort_if(!$device->tenant_id, 403, 'الشاشة غير مربوطة بعد.');

        // Update last_seen timestamp to track online status
        $device->update([
            'last_ip' => request()->ip(),
            'last_seen_at' => now()
        ]);

        // فلترة بالـ tenant_id
        $tenantId = $device->tenant_id;

        $current = WaitingQueue::with('customer')
            ->where('tenant_id', $tenantId)
            ->where('status', 'serving')
            ->whereDate('arrival_time', today())
            ->orderByDesc('priority_score')
            ->orderBy('arrival_time')
            ->first();

        // Get all waiting customers (simplified query for better debugging)
        $next = WaitingQueue::with('customer')
            ->where('tenant_id', $tenantId)
            ->where('status', 'waiting')
            ->when($current, fn($q) => $q->where('id', '!=', $current->id))
            ->orderByDesc('priority_score')
            ->orderBy('created_at') // Show oldest customers first (ascending)
            ->get();

        // Filter and take appointments and walk-ins  
        $appointments = $next->filter(fn($q) => $q->type === 'appointment')
                            ->take(5) // Show more for testing
                            ->values();
        $walkins     = $next->filter(fn($q) => $q->type === 'walk-in')
                            ->take(5) // Show more for testing
                            ->values();

        return response()->json([
            'current'      => $current,
            'appointments' => $appointments,
            'walkins'      => $walkins,
        ]);
    }
    
    /**
     * Heartbeat endpoint for displays to report their online status
     */
    public function heartbeat(string $device_token)
    {
        $device = $this->resolveDeviceOrAbort($device_token);
        
        // Update last seen timestamp
        $device->update([
            'last_ip' => request()->ip(),
            'last_seen_at' => now()
        ]);
        
        return response()->json([
            'success' => true,
            'timestamp' => now()->toISOString()
        ]);
    }
    
    /**
     * Show all displays for the current tenant
     */
    public function list(Request $request)
    {
        $user = auth()->user();
        
        // Get all displays for the current tenant
        $displays = DisplayDevice::where('tenant_id', $user->tenant_id)
            ->orderBy('created_at', 'desc')
            ->get();
            
        $displaysWithStatus = $displays->map(function ($display) {
            // Determine status based on authorization and last activity
            $status = 'disconnected';
            $statusColor = 'danger';
            $statusText = __('displays.disconnected');
            
            if ($display->is_authorized === 'oui') { // French: 'oui' = yes
                // Check if device is active (has activity within last 2 minutes)
                $lastSeen = $display->last_seen_at;
                $isOnline = $lastSeen && $lastSeen->gt(now()->subMinutes(2));
                
                if ($isOnline) {
                    $status = 'online';
                    $statusColor = 'success';
                    $statusText = __('displays.online');
                } else {
                    $status = 'offline';
                    $statusColor = 'warning';
                    $statusText = __('displays.offline');
                }
            } elseif ($display->is_authorized === 'pending') {
                $status = 'pending';
                $statusColor = 'info';
                $statusText = __('displays.pending_authorization');
            } else { // 'non' or any other value
                $status = 'disconnected';
                $statusColor = 'danger';
                $statusText = __('displays.not_authorized');
            }
            
            // Get or create short link for this display
            $shortLink = ShortLink::where('tenant_id', $display->tenant_id)
                ->where('usage_type', 'display_pairing')
                ->where('metadata->display_id', $display->id)
                ->active()
                ->first();
                
            if (!$shortLink) {
                // Create short link if it doesn't exist
                $pairingUrl = route('display.pairing', $display->device_token);
                try {
                    $shortLink = ShortLink::createForDisplay(
                        $pairingUrl,
                        $display->tenant_id,
                        $display->id,
                        auth()->user()->email ?? 'system'
                    );
                } catch (\Exception $e) {
                    // If short link creation fails, continue without it
                    $shortLink = null;
                }
            }
            
            return [
                'id' => $display->id,
                'name' => $display->name ?: __('displays.unnamed_display'),
                'device_token' => $display->device_token,
                'last_ip' => $display->last_ip,
                'is_authorized' => $display->is_authorized,
                'paired_at' => $display->paired_at,
                'created_at' => $display->created_at,
                'updated_at' => $display->updated_at,
                'status' => $status,
                'status_color' => $statusColor,
                'status_text' => $statusText,
                'display_url' => route('display.index', $display->device_token),
                'pairing_url' => route('display.pairing', $display->device_token),
                'short_link' => $shortLink ? [
                    'id' => $shortLink->id,
                    'short_url' => $shortLink->short_url,
                    'short_code' => $shortLink->short_code,
                    'click_count' => $shortLink->click_count,
                    'expires_at' => $shortLink->expires_at
                ] : null,
            ];
        });
        
        // Statistics
        $stats = [
            'total' => $displays->count(),
            'online' => $displaysWithStatus->where('status', 'online')->count(),
            'offline' => $displaysWithStatus->where('status', 'offline')->count(),
            'pending' => $displaysWithStatus->where('status', 'pending')->count(),
            'disconnected' => $displaysWithStatus->where('status', 'disconnected')->count(),
        ];
        
        return view('displays.index', compact('displaysWithStatus', 'stats'));
    }
    
    /**
     * Create a new display device
     */
    public function create(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255'
        ]);
        
        $display = DisplayDevice::create([
            'name' => $request->name,
            'device_token' => Str::random(40), // Use 40 chars like in the model
            'tenant_id' => auth()->user()->tenant_id,
            'is_authorized' => 'non', // French: 'non' = no (default value)
            'last_ip' => $request->ip()
        ]);
        
        // Generate short link for pairing
        $pairingUrl = route('display.pairing', $display->device_token);
        $shortLink = ShortLink::createForDisplay(
            $pairingUrl,
            auth()->user()->tenant_id,
            $display->id,
            auth()->user()->email
        );
        
        if ($request->expectsJson()) {
            return response()->json([
                'success' => true,
                'message' => __('settings.display_created'),
                'display' => $display,
                'pairing_url' => $pairingUrl,
                'short_url' => $shortLink->short_url,
                'short_code' => $shortLink->short_code
            ]);
        }
        
        return redirect()->route('displays.index')->with('success', __('settings.display_created'));
    }
    
    /**
     * Get or create short link for display pairing
     */
    public function getShortLink(DisplayDevice $display)
    {
        // Check if short link already exists
        $shortLink = ShortLink::where('tenant_id', $display->tenant_id)
            ->where('usage_type', 'display_pairing')
            ->where('metadata->display_id', $display->id)
            ->active()
            ->first();
            
        if (!$shortLink) {
            // Create new short link
            $pairingUrl = route('display.pairing', $display->device_token);
            $shortLink = ShortLink::createForDisplay(
                $pairingUrl,
                $display->tenant_id,
                $display->id,
                auth()->user()->email ?? 'system'
            );
        }
        
        return $shortLink;
    }
    
    /**
     * Delete a display device
     */
    public function destroy(Request $request, $id)
    {
        try {
            $display = DisplayDevice::where('id', $id)
                ->where('tenant_id', auth()->user()->tenant_id)
                ->first();
                
            if (!$display) {
                return response()->json([
                    'success' => false,
                    'message' => __('displays.display_not_found')
                ], 404);
            }
            
            // Delete associated short links
            ShortLink::where('tenant_id', $display->tenant_id)
                ->where('usage_type', 'display_pairing')
                ->where('metadata->display_id', $display->id)
                ->delete();
                
            // Delete the display
            $displayName = $display->name;
            $display->delete();
            
            if ($request->expectsJson()) {
                return response()->json([
                    'success' => true,
                    'message' => __('displays.display_deleted_successfully', ['name' => $displayName])
                ]);
            }
            
            return redirect()->route('displays.index')
                ->with('success', __('displays.display_deleted_successfully', ['name' => $displayName]));
                
        } catch (\Exception $e) {
            if ($request->expectsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => __('displays.delete_error')
                ], 500);
            }
            
            return redirect()->route('displays.index')
                ->with('error', __('displays.delete_error'));
        }
    }
    
    /**
     * Update a display device
     */
    public function update(Request $request, $id)
    {
        $request->validate([
            'name' => 'required|string|max:255'
        ]);
        
        try {
            $display = DisplayDevice::where('id', $id)
                ->where('tenant_id', auth()->user()->tenant_id)
                ->first();
                
            if (!$display) {
                return response()->json([
                    'success' => false,
                    'message' => __('displays.display_not_found')
                ], 404);
            }
            
            $oldName = $display->name;
            $display->name = $request->name;
            $display->save();
            
            if ($request->expectsJson()) {
                return response()->json([
                    'success' => true,
                    'message' => __('displays.display_updated_successfully', ['old' => $oldName, 'new' => $request->name]),
                    'display' => $display
                ]);
            }
            
            return redirect()->route('displays.index')
                ->with('success', __('displays.display_updated_successfully', ['old' => $oldName, 'new' => $request->name]));
                
        } catch (\Exception $e) {
            if ($request->expectsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => __('displays.update_error')
                ], 500);
            }
            
            return redirect()->route('displays.index')
                ->with('error', __('displays.update_error'));
        }
    }
}
