<?php

namespace App\Http\Controllers;

use App\Models\DisplayDevice;
use Illuminate\Http\Request;
use SimpleSoftwareIO\QrCode\Facades\QrCode;
use App\Events\NewDisplay;

class DisplayPairController extends Controller
{
    protected function getDeviceOr404(string $device_token): DisplayDevice
    {
        $device = DisplayDevice::where('device_token', $device_token)->first();
        abort_if(!$device, 404);
        return $device;
    }

    // شاشة التلفاز: إظهار QR ليتم مسحه من الهاتف
    public function showPairing(string $device_token)
    {
        $device = $this->getDeviceOr404($device_token);

        if ($device->is_authorized == 'oui') {
           return redirect('/display/'.$device->device_token);
        }

        // الرابط الذي سيفتحه صاحب المحل على هاتفه (يتطلب auth)
        $pairUrl = route('display.pair.form', $device->device_token);

        $qrSvg = QrCode::size(280)->margin(1)->generate($pairUrl);

        return view('display.pairing', [
            'device' => $device,
            'qrSvg'  => $qrSvg,
            'pairUrl'=> $pairUrl, // للنسخ اليدوي كخطة بديلة
            'tenantId'=> auth()->user()->tenant_id,
        ]);
    }

    // صفحة الهاتف بعد تسجيل الدخول: نموذج تأكيد الربط
    public function pairForm(string $device_token)
    {
        $device = $this->getDeviceOr404($device_token);

        // لو مربوط أصلاً، رجّعه للوحة التحكم
        if ($device->tenant_id && $device->is_authorized == 'oui' ) {
            return redirect()->route('dashboard')->with('info', 'هذا الجهاز مربوط مسبقًا.');
        }

        return view('display.pair-confirm', [
            'device' => $device,
            'user'   => auth()->user(),
        ]);
    }

    // تنفيذ الربط
    public function pairStore(Request $request, string $device_token)
    {
        $device = $this->getDeviceOr404($device_token);

        // إن كان مربوطاً بالفعل
        if ($device->tenant_id && $device->is_authorized == 'oui') {
            return redirect()->route('dashboard')->with('info', 'هذا الجهاز مربوط مسبقًا.');
        }

        $device->tenant_id = auth()->user()->tenant_id;
        $device->paired_at = now();
        $device->last_ip   = $request->ip();
        $device->name      = $request->input('name'); // اسم اختياري للشاشة
        $device->is_authorized = 'oui';
        $device->save();

        broadcast(new NewDisplay(auth()->user()->tenant_id));

        return redirect()->route('dashboard')->with('success', 'تم ربط الجهاز بنجاح.');
    }

    // إلغاء الربط من لوحة التحكم
    public function unpair(string $device_token)
    {
        $device = $this->getDeviceOr404($device_token);

        // تأكد أن المستخدم له نفس الـ tenant
        abort_if($device->tenant_id !== auth()->user()->tenant_id, 403);

        $device->tenant_id = null;
        $device->paired_at = null;
        $device->is_authorized = 'non';
        $device->save();

        return back()->with('success', 'تم إلغاء ربط الجهاز.');
    }
}
