<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Patient;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;
use Carbon\Carbon;

class PatientAuthController extends Controller
{
    /**
     * Show patient registration form
     */
    public function showRegistrationForm()
    {
        return view('patient.auth.register');
    }

    /**
     * Handle patient registration
     */
    public function register(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'contact_method' => 'required|in:email,phone',
            'email' => 'required_if:contact_method,email|email|unique:patients,email',
            'phone' => 'required_if:contact_method,phone|string|unique:patients,phone',
            'password' => 'nullable|string|min:6|confirmed',
            'date_of_birth' => 'nullable|date',
            'preferred_language' => 'in:ar,en,fr',
        ]);

        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        $patientData = [
            'name' => $request->name,
            'preferred_language' => $request->preferred_language ?? 'ar',
            'date_of_birth' => $request->date_of_birth,
            'is_active' => true,
        ];

        if ($request->contact_method === 'email') {
            $patientData['email'] = $request->email;
            if ($request->password) {
                $patientData['password'] = Hash::make($request->password);
            }
        } else {
            $patientData['phone'] = $request->phone;
            // For phone registration, we'll verify via SMS later
        }

        try {
            $patient = Patient::create($patientData);

            // Auto-verify for now (in production, send verification email/SMS)
            if ($request->contact_method === 'email') {
                $patient->email_verified_at = now();
            } else {
                $patient->phone_verified_at = now();
            }
            $patient->save();

            // Log in the patient
            Auth::guard('patient')->login($patient);

            return redirect()->route('patient.dashboard')
                ->with('success', __('messages.registration_successful'));

        } catch (\Exception $e) {
            return back()->withErrors(['general' => __('messages.registration_failed')])
                ->withInput();
        }
    }

    /**
     * Show patient login form
     */
    public function showLoginForm()
    {
        return view('patient.auth.login');
    }

    /**
     * Handle patient login
     */
    public function login(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'login_method' => 'required|in:email,phone',
            'email' => 'required_if:login_method,email|email',
            'phone' => 'required_if:login_method,phone|string',
            'password' => 'required_if:login_method,email|string',
            'verification_code' => 'required_if:login_method,phone|string',
        ]);

        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        try {
            if ($request->login_method === 'email') {
                // Email + password login
                $credentials = [
                    'email' => $request->email,
                    'password' => $request->password,
                    'is_active' => true
                ];

                if (Auth::guard('patient')->attempt($credentials, $request->remember)) {
                    $request->session()->regenerate();
                    return redirect()->intended(route('patient.dashboard'));
                }
            } else {
                // Phone + verification code login
                // In production, verify the SMS code here
                $patient = Patient::where('phone', $request->phone)
                    ->where('is_active', true)
                    ->first();

                if ($patient && $this->verifyPhoneCode($request->phone, $request->verification_code)) {
                    Auth::guard('patient')->login($patient, $request->remember);
                    $request->session()->regenerate();
                    return redirect()->intended(route('patient.dashboard'));
                }
            }

            return back()->withErrors([
                'general' => __('messages.invalid_credentials')
            ])->withInput();

        } catch (\Exception $e) {
            return back()->withErrors(['general' => __('messages.login_failed')])
                ->withInput();
        }
    }

    /**
     * Send verification code to phone (placeholder)
     */
    public function sendVerificationCode(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'phone' => 'required|string'
        ]);

        if ($validator->fails()) {
            return response()->json(['success' => false, 'errors' => $validator->errors()]);
        }

        // Check if patient exists
        $patient = Patient::where('phone', $request->phone)->where('is_active', true)->first();

        if (!$patient) {
            return response()->json([
                'success' => false,
                'message' => __('messages.phone_not_registered')
            ]);
        }

        // In production, send actual SMS here
        // For now, we'll just return success
        $verificationCode = rand(1000, 9999);
        
        // Store verification code in session for demo
        session(['phone_verification_code' => $verificationCode]);
        session(['phone_verification_phone' => $request->phone]);
        session(['phone_verification_expires' => now()->addMinutes(5)]);

        return response()->json([
            'success' => true,
            'message' => __('messages.verification_code_sent'),
            'code' => $verificationCode // Remove this in production
        ]);
    }

    /**
     * Verify phone code (placeholder)
     */
    private function verifyPhoneCode($phone, $code)
    {
        // In production, verify against your SMS service
        $sessionCode = session('phone_verification_code');
        $sessionPhone = session('phone_verification_phone');
        $expiresAt = session('phone_verification_expires');

        if (!$sessionCode || !$sessionPhone || !$expiresAt) {
            return false;
        }

        if (now()->isAfter($expiresAt)) {
            return false; // Code expired
        }

        if ($sessionPhone !== $phone || $sessionCode != $code) {
            return false;
        }

        // Clear session data
        session()->forget(['phone_verification_code', 'phone_verification_phone', 'phone_verification_expires']);
        
        return true;
    }

    /**
     * Patient logout
     */
    public function logout(Request $request)
    {
        Auth::guard('patient')->logout();
        $request->session()->invalidate();
        $request->session()->regenerateToken();

        return redirect()->route('patient.login')
            ->with('success', __('messages.logout_successful'));
    }

    /**
     * Show quick join form for a specific tenant
     */
    public function showQuickJoinForm($uuid)
    {
        $tenant = \App\Models\Tenant::where('uuid', $uuid)
            ->with(['visitTypes'])
            ->firstOrFail();
        
        return view('patient.quick-join', compact('tenant'));
    }

    /**
     * Quick join - create join request for patients to join queue
     */
    public function quickJoin(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'phone' => 'required|string',
            'tenant_uuid' => 'required|string|exists:tenants,uuid',
        ]);

        if ($validator->fails()) {
            return response()->json(['success' => false, 'errors' => $validator->errors()]);
        }

        try {
            // Find or create patient
            $patient = Patient::where('phone', $request->phone)->first();
            
            if (!$patient) {
                $patient = Patient::create([
                    'name' => $request->name,
                    'phone' => $request->phone,
                    'phone_verified_at' => now(), // Quick join auto-verifies
                    'is_active' => true,
                ]);
            } else {
                // Update patient name in case it changed
                $patient->update(['name' => $request->name]);
            }

            // Find tenant
            $tenant = \App\Models\Tenant::where('uuid', $request->tenant_uuid)->first();
            
            // Check if patient already has a pending request for this tenant
            $existingRequest = \App\Models\JoinRequest::withoutGlobalScope(\App\Models\Scopes\TenantScope::class)
                ->where('patient_id', $patient->id)
                ->where('tenant_id', $tenant->id)
                ->where('status', 'pending')
                ->first();
                
            if ($existingRequest) {
                return response()->json([
                    'success' => false,
                    'message' => __('لديك طلب انتظار معلق بالفعل لهذا المكان')
                ]);
            }
            
            // Check if patient is already in queue
            $currentQueue = $patient->queueEntries()
                ->where('tenant_id', $tenant->id)
                ->whereIn('status', ['waiting', 'serving'])
                ->first();
                
            if ($currentQueue) {
                return response()->json([
                    'success' => false,
                    'message' => __('أنت في قائمة الانتظار بالفعل لهذا المكان')
                ]);
            }
            
            // Create join request
            $joinRequest = \App\Models\JoinRequest::create([
                'patient_id' => $patient->id,
                'tenant_id' => $tenant->id,
                'status' => 'pending',
                'requested_at' => now(),
            ]);
            
            // Log in the patient temporarily
            Auth::guard('patient')->login($patient);

            return response()->json([
                'success' => true,
                'message' => __('تم إرسال طلب الانتظار بنجاح. سيتم مراجعة طلبك من قبل العيادة.'),
                'redirect_url' => route('patient.dashboard')
            ]);

        } catch (\Exception $e) {
            \Log::error('Quick join failed: ' . $e->getMessage(), [
                'request_data' => $request->all(),
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => __('فشل في إرسال طلب الانتظار. يرجى المحاولة مرة أخرى.')
            ]);
        }
    }

    /**
     * Calculate average wait time for a tenant
     */
    private function calculateAverageWaitTime($tenantId)
    {
        $avgTime = \App\Models\WaitingQueue::where('tenant_id', $tenantId)
            ->whereNotNull('served_at')
            ->whereNotNull('completed_at')
            ->where('completed_at', '>=', now()->subDays(7))
            ->selectRaw('AVG(TIMESTAMPDIFF(MINUTE, arrival_time, served_at)) as avg_minutes')
            ->first();
        
        return $avgTime->avg_minutes ?? 15;
    }
}
