<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Patient;
use App\Models\Tenant;
use App\Models\WaitingQueue;
use App\Models\JoinRequest;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class PatientController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth:patient');
    }

    /**
     * Patient dashboard
     */
    public function dashboard()
    {
        $patient = Auth::guard('patient')->user();

        // Get current queue entries
        $currentQueues = $patient->queueEntries()
            ->with(['tenant'])
            ->whereIn('status', ['waiting', 'serving'])
            ->get();

        // Add position/appointment information for each queue
        foreach ($currentQueues as $queue) {
            $positionData = $patient->getQueuePositionAt($queue->tenant_id, $queue->id);
            if ($positionData) {
                $queue->queue_type = $positionData['type'];
                if ($positionData['type'] === 'walk-in') {
                    $queue->position = $positionData['position'] ?? null;
                    $queue->total_walkins_waiting = $positionData['total_walkins_waiting'] ?? 0;
                } else {
                    $queue->appointment_time = $positionData['appointment_time'] ?? null;
                    $queue->is_appointment_due = $positionData['is_appointment_due'] ?? false;
                }
            }
        }


        // Get recent visits history
        $recentVisits = $patient->visits()
            ->with(['tenant'])
            ->limit(5)
            ->get();

        // Get favorite tenants
        $favoriteTenants = $patient->favoriteTenants()
            ->with(['visitTypes'])
            ->get();

        return view('patient.dashboard', compact(
            'patient',
            'currentQueues',
            'recentVisits',
            'favoriteTenants'
        ));
    }

    /**
     * Search for tenants/places
     */
    public function search(Request $request)
    {
        $query = $request->get('q', '');
        $location = $request->get('location', '');

        $tenants = Tenant::query()
            ->when($query, function ($q) use ($query) {
                return $q->where('name', 'like', "%{$query}%")
                        ->orWhere('address', 'like', "%{$query}%")
                        ->orWhere('email', 'like', "%{$query}%")
                        ->orWhereHas('visitTypes', function($vt) use ($query) {
                            $vt->where('name', 'like', "%{$query}%");
                        });
            })
            ->when($location, function ($q) use ($location) {
                return $q->where('address', 'like', "%{$location}%");
            })
            ->with(['visitTypes'])
            ->orderBy('name')
            ->paginate(12);

        return view('patient.search', compact('tenants', 'query', 'location'));
    }

    /**
     * Show tenant details and request status
     */
    public function showTenant($uuid)
    {
        $tenant = Tenant::where('uuid', $uuid)
            ->with(['visitTypes'])
            ->firstOrFail();

        $patient = Auth::guard('patient')->user();

        // Check if patient is currently in queue
        $currentQueue = $patient->queueEntries()
            ->where('tenant_id', $tenant->id)
            ->whereIn('status', ['waiting', 'serving'])
            ->first();

        $positionData = null;
        if ($currentQueue) {
            $positionData = $patient->getQueuePositionAt($tenant->id, $currentQueue->id);
        }

        // Check for pending join request
        $pendingRequest = JoinRequest::withoutGlobalScope(\App\Models\Scopes\TenantScope::class)
            ->where('patient_id', $patient->id)
            ->where('tenant_id', $tenant->id)
            ->where('status', 'pending')
            ->first();

        // Check if tenant is in favorites
        $isFavorite = $patient->hasFavoriteTenant($tenant->id);

        return view('patient.tenant', compact(
            'tenant',
            'currentQueue',
            'positionData',
            'pendingRequest',
            'isFavorite'
        ));
    }

    /**
     * Create join request (instead of directly joining queue)
     */
    public function joinQueue(Request $request, $uuid)
    {
        $tenant = Tenant::where('uuid', $uuid)->firstOrFail();
        $patient = Auth::guard('patient')->user();

        // Check if patient already has a pending request for this tenant
        $existingRequest = JoinRequest::withoutGlobalScope(\App\Models\Scopes\TenantScope::class)
            ->where('patient_id', $patient->id)
            ->where('tenant_id', $tenant->id)
            ->where('status', 'pending')
            ->first();

        if ($existingRequest) {
            return response()->json([
                'success' => false,
                'message' => __('لديك طلب انتظار معلق بالفعل لهذا المكان')
            ], 400);
        }

        // Check if patient is already in queue
        $currentQueue = $patient->queueEntries()
            ->where('tenant_id', $tenant->id)
            ->whereIn('status', ['waiting', 'serving'])
            ->first();

        if ($currentQueue) {
            return response()->json([
                'success' => false,
                'message' => __('أنت في قائمة الانتظار بالفعل لهذا المكان')
            ], 400);
        }

        try {
            // Create join request instead of direct queue entry
            $joinRequest = JoinRequest::create([
                'patient_id' => $patient->id,
                'tenant_id' => $tenant->id,
                'status' => 'pending',
                'requested_at' => now(),
            ]);

            // TODO: Add notification to staff about new join request

            return response()->json([
                'success' => true,
                'message' => __('تم إرسال طلب الانتظار بنجاح. سيتم مراجعة طلبك من قبل العيادة.'),
                'request_id' => $joinRequest->uuid,
                'status' => 'pending'
            ]);

        } catch (\Exception $e) {
            \Log::error('Failed to create join request: ' . $e->getMessage(), [
                'patient_id' => $patient->id,
                'tenant_id' => $tenant->id,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return response()->json([
                'success' => false,
                'message' => __('فشل في إرسال طلب الانتظار. يرجى المحاولة مرة أخرى.') . ': ' . $e->getMessage()
            ], 400);
        }
    }

    /**
     * Leave queue
     */
    public function leaveQueue($uuid)
    {
        $tenant = Tenant::where('uuid', $uuid)->firstOrFail();
        $patient = Auth::guard('patient')->user();

        $queueEntry = $patient->queueEntries()
            ->where('tenant_id', $tenant->id)
            ->whereIn('status', ['waiting', 'serving'])
            ->first();

        if (!$queueEntry) {
            return response()->json([
                'success' => false,
                'message' => __('messages.not_in_queue')
            ], 400);
        }

        try {
            $queueEntry->update([
                'status' => 'cancelled',
                'completed_at' => now(),
                'notes' => ($queueEntry->notes ?? '') . ' - Cancelled by patient'
            ]);

            // Trigger WebSocket update
            broadcast(new \App\Events\QueueUpdated($tenant->id));

            return response()->json([
                'success' => true,
                'message' => __('messages.left_queue_successfully')
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => __('messages.leave_queue_failed')
            ], 400);
        }
    }

    /**
     * Get live queue status for patient (including join requests)
     */
    public function getQueueStatus($uuid)
    {
        $tenant = Tenant::where('uuid', $uuid)->firstOrFail();
        $patient = Auth::guard('patient')->user();

        // First check for pending join request
        $pendingRequest = JoinRequest::withoutGlobalScope(\App\Models\Scopes\TenantScope::class)
            ->where('patient_id', $patient->id)
            ->where('tenant_id', $tenant->id)
            ->where('status', 'pending')
            ->first();

        if ($pendingRequest) {
            return response()->json([
                'has_request' => true,
                'request_status' => 'pending',
                'message' => __('طلبك قيد المراجعة'),
                'requested_at' => $pendingRequest->requested_at->format('H:i'),
                'request_id' => $pendingRequest->uuid
            ]);
        }

        // Check if request was rejected
        $rejectedRequest = JoinRequest::withoutGlobalScope(\App\Models\Scopes\TenantScope::class)
            ->where('patient_id', $patient->id)
            ->where('tenant_id', $tenant->id)
            ->where('status', 'rejected')
            ->latest('responded_at')
            ->first();

        // Then check for active queue entry
        $positionData = $patient->getQueuePositionAt($tenant->id);

        if (!$positionData && !$rejectedRequest) {
            return response()->json([
                'in_queue' => false,
                'has_request' => false,
                'message' => __('لست في قائمة الانتظار')
            ]);
        }

        if ($rejectedRequest && !$positionData) {
            return response()->json([
                'in_queue' => false,
                'has_request' => false,
                'request_rejected' => true,
                'rejection_reason' => $rejectedRequest->notes,
                'rejected_at' => $rejectedRequest->responded_at->format('H:i'),
                'message' => __('تم رفض طلبك')
            ]);
        }

        if ($positionData) {
            $queueEntry = $positionData['queue_entry'];

            $response = [
                'in_queue' => true,
                'status' => $queueEntry->status,
                'type' => $positionData['type'],
                'arrival_time' => $queueEntry->arrival_time->format('H:i'),
            ];

            if ($positionData['type'] === 'walk-in') {
                $response['position'] = $positionData['position'];
                $response['total_walkins_waiting'] = $positionData['total_walkins_waiting'];
            } else {
                $response['appointment_time'] = $positionData['appointment_time']->format('d/m/Y H:i');
                $response['is_appointment_due'] = $positionData['is_appointment_due'];
            }

            return response()->json($response);
        }

        return response()->json([
            'in_queue' => false,
            'has_request' => false,
            'message' => __('لست في قائمة الانتظار')
        ]);
    }

    /**
     * Toggle favorite tenant
     */
    public function toggleFavorite($uuid)
    {
        $tenant = Tenant::where('uuid', $uuid)->firstOrFail();
        $patient = Auth::guard('patient')->user();

        try {
            if ($patient->hasFavoriteTenant($tenant->id)) {
                $patient->removeFavoriteTenant($tenant->id);
                $isFavorite = false;
                $message = __('messages.removed_from_favorites');
            } else {
                $patient->addFavoriteTenant($tenant->id);
                $isFavorite = true;
                $message = __('messages.added_to_favorites');
            }

            return response()->json([
                'success' => true,
                'is_favorite' => $isFavorite,
                'message' => $message
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => __('messages.action_failed')
            ], 400);
        }
    }

    /**
     * Patient profile
     */
    public function profile()
    {
        $patient = Auth::guard('patient')->user();
        return view('patient.profile', compact('patient'));
    }

    /**
     * Update patient profile
     */
    public function updateProfile(Request $request)
    {
        $patient = Auth::guard('patient')->user();

        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'nullable|email|unique:patients,email,' . $patient->id,
            'phone' => 'nullable|string|unique:patients,phone,' . $patient->id,
            'date_of_birth' => 'nullable|date',
            'address' => 'nullable|string',
            'emergency_contact_name' => 'nullable|string|max:255',
            'emergency_contact_phone' => 'nullable|string',
            'preferred_language' => 'in:ar,en,fr',
        ]);

        try {
            $patient->update($request->only([
                'name', 'email', 'phone', 'date_of_birth', 'address',
                'emergency_contact_name', 'emergency_contact_phone', 'preferred_language'
            ]));

            return back()->with('success', __('messages.profile_updated_successfully'));

        } catch (\Exception $e) {
            return back()->withErrors(['general' => __('messages.profile_update_failed')]);
        }
    }

    /**
     * Visit history
     */
    public function visitHistory()
    {
        $patient = Auth::guard('patient')->user();

        $visits = $patient->visits()
            ->with(['tenant'])
            ->paginate(15);

        return view('patient.history', compact('visits'));
    }

    /**
     * Favorite tenants
     */
    public function favorites()
    {
        $patient = Auth::guard('patient')->user();

        $favorites = $patient->favoriteTenants()
            ->with(['visitTypes'])
            ->paginate(12);

        return view('patient.favorites', compact('favorites'));
    }

}
