<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Models\QrLoginToken;
use App\Models\User;
use App\Models\ActivityLog;
use SimpleSoftwareIO\QrCode\Facades\QrCode;
use Carbon\Carbon;
use Illuminate\Support\Facades\Hash;

class QrLoginController extends Controller
{
    /**
     * Generate a new QR code for login
     */
    public function generateQrCode(Request $request)
    {
        // Clean up expired tokens
        $this->cleanupExpiredTokens();
        
        // Generate new token
        $token = QrLoginToken::generateToken();
        $expiresAt = now()->addMinutes(5); // QR code expires in 5 minutes
        
        $qrToken = QrLoginToken::create([
            'token' => $token,
            'expires_at' => $expiresAt,
            'ip_address' => $request->ip(),
            'user_agent' => $request->userAgent()
        ]);
        
        // Create QR code URL - this will be the mobile login URL
        $qrLoginUrl = route('qr.login.scan', ['token' => $token]);
        
        // Generate QR code SVG
        $qrCode = QrCode::size(200)
            ->margin(2)
            ->generate($qrLoginUrl);
            
        // Convert to string if it's not already
        if (is_object($qrCode)) {
            $qrCode = (string) $qrCode;
        }
            
        return response()->json([
            'success' => true,
            'token' => $token,
            'qr_code' => $qrCode,
            'expires_at' => $expiresAt->toISOString(),
            'expires_in' => 300 // 5 minutes in seconds
        ]);
    }
    
    /**
     * Show QR scan page (mobile view)
     */
    public function showScanPage($token)
    {
        $qrToken = QrLoginToken::where('token', $token)
            ->where('status', 'pending')
            ->first();
            
        if (!$qrToken || $qrToken->isExpired()) {
            return view('auth.qr-expired');
        }
        
        return view('auth.qr-scan', compact('qrToken'));
    }
    
    /**
     * Handle QR scan authentication (mobile)
     */
    public function authenticateQrScan(Request $request, $token)
    {
        $request->validate([
            'email' => 'required|email',
            'password' => 'required'
        ]);
        
        $qrToken = QrLoginToken::where('token', $token)
            ->where('status', 'pending')
            ->first();
            
        if (!$qrToken || $qrToken->isExpired()) {
            return response()->json([
                'success' => false,
                'message' => 'QR code has expired. Please generate a new one.'
            ], 400);
        }
        
        // Verify user credentials
        $user = User::where('email', $request->email)->first();
        
        if (!$user || !Hash::check($request->password, $user->password)) {
            return response()->json([
                'success' => false,
                'message' => 'Invalid credentials.'
            ], 401);
        }
        
        if (!$user->is_active) {
            return response()->json([
                'success' => false,
                'message' => 'Your account is not active. Please contact the administrator.'
            ], 403);
        }
        
        // Mark token as scanned with user info
        $qrToken->markAsScanned($user->id);
        
        return response()->json([
            'success' => true,
            'message' => 'QR code scanned successfully. Please check your computer to complete login.',
            'user_name' => $user->name
        ]);
    }
    
    /**
     * Check QR login status (desktop polling)
     */
    public function checkQrStatus($token)
    {
        $qrToken = QrLoginToken::where('token', $token)->first();
        
        if (!$qrToken) {
            return response()->json([
                'success' => false,
                'status' => 'invalid',
                'message' => 'Invalid QR token.'
            ]);
        }
        
        if ($qrToken->isExpired()) {
            $qrToken->markAsExpired();
            return response()->json([
                'success' => false,
                'status' => 'expired',
                'message' => 'QR code has expired.'
            ]);
        }
        
        return response()->json([
            'success' => true,
            'status' => $qrToken->status,
            'user_id' => $qrToken->user_id,
            'user_name' => $qrToken->user ? $qrToken->user->name : null,
            'scanned_at' => $qrToken->scanned_at ? $qrToken->scanned_at->toISOString() : null
        ]);
    }
    
    /**
     * Complete QR login (desktop)
     */
    public function completeQrLogin(Request $request, $token)
    {
        $qrToken = QrLoginToken::where('token', $token)
            ->where('status', 'scanned')
            ->first();
            
        if (!$qrToken || $qrToken->isExpired() || !$qrToken->user_id) {
            return response()->json([
                'success' => false,
                'message' => 'Invalid or expired QR login session.'
            ], 400);
        }
        
        $user = User::find($qrToken->user_id);
        
        if (!$user || !$user->is_active) {
            return response()->json([
                'success' => false,
                'message' => 'User account is not active.'
            ], 403);
        }
        
        // Log the user in
        Auth::login($user);
        
        // Mark token as authenticated
        $qrToken->markAsAuthenticated();
        
        // Log the login activity
        ActivityLog::create([
            'user_id' => $user->id,
            'action' => 'qr_login',
            'description' => 'QR code login successful',
            'ip_address' => $request->ip(),
            'user_agent' => $request->userAgent(),
        ]);
        
        $request->session()->regenerate();
        
        return response()->json([
            'success' => true,
            'message' => 'Login successful!',
            'redirect_url' => '/dashboard'
        ]);
    }
    
    /**
     * Clean up expired tokens
     */
    private function cleanupExpiredTokens()
    {
        QrLoginToken::where('expires_at', '<', now())
            ->where('status', '!=', 'expired')
            ->update(['status' => 'expired']);
    }
}
