<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use App\Models\Permission;
use App\Models\User;
use Illuminate\Support\Facades\Auth;

class RolePermissionController extends Controller
{
    public function create()
    {
        $tenant = Auth::user()->tenant;
        $planId = $tenant->subscription_plan_id;

        // Check tenant limits first
        if ($tenant->id != 1) {
            if ($limitPage = $this->checkTenantLimit(auth()->user()->tenant_id)) {
                return $limitPage;
            }
        }

        // Get available permission groups for creating new roles
        $permissionGroups = $this->getAvailablePermissionGroups($planId, $tenant->id);
        
        // Get existing roles for this tenant
        $existingRoles = DB::table('roles')
            ->where('tenant_id', $tenant->id)
            ->get();

        return view('roles.create_with_user_redesigned', compact('permissionGroups', 'existingRoles'));
    }
    
    /**
     * Get available permission groups based on tenant's plan
     */
    private function getAvailablePermissionGroups($planId, $tenantId)
    {
        if ($tenantId == 1) {
            // Super admin sees all groups
            return Permission::distinct()
                ->pluck('group')
                ->filter()
                ->sort()
                ->values();
        } else {
            // Regular tenants see groups based on their plan
            return Permission::select('permissions.group')
                ->join('permission_plan', 'permissions.id', '=', 'permission_plan.permission_id')
                ->where('permission_plan.plan_id', $planId)
                ->distinct()
                ->pluck('group')
                ->filter()
                ->sort()
                ->values();
        }
    }
    
    /**
     * Get all permissions for a specific group
     */
    private function getPermissionsByGroup($groupName, $planId, $tenantId)
    {
        if ($tenantId == 1) {
            // Super admin gets all permissions in the group
            return Permission::where('group', $groupName)->pluck('id');
        } else {
            // Regular tenants get permissions based on their plan
            return Permission::select('permissions.id')
                ->join('permission_plan', 'permissions.id', '=', 'permission_plan.permission_id')
                ->where('permission_plan.plan_id', $planId)
                ->where('permissions.group', $groupName)
                ->pluck('id');
        }
    }

    public function store(Request $request)
    {
        // Define validation rules based on user_type
        $rules = [
            'user_type' => 'required|in:existing_role,new_role',
            'user_name' => 'required|string|max:255',
            'user_email' => 'required|email|unique:users,email',
            'user_password' => 'required|string|min:6',
        ];
        
        // Add conditional validation rules based on user_type
        if ($request->user_type === 'existing_role') {
            $rules['existing_role_id'] = 'required|exists:roles,id';
        } elseif ($request->user_type === 'new_role') {
            $rules['role_name'] = 'required|string|max:255';
            $rules['permission_groups'] = 'required|array|min:1';
            $rules['permission_groups.*'] = 'string';
        }
        
        $request->validate($rules);

        $tenant = Auth::user()->tenant;
        $tenantId = $tenant->id;
        $planId = $tenant->subscription_plan_id;

        $result = DB::transaction(function () use ($request, $tenantId, $planId) {
            // 1️⃣ Determine Role ID
            if ($request->user_type === 'existing_role') {
                // Use existing role
                $roleId = $request->existing_role_id;
            } else {
                // Create new role
                $roleId = DB::table('roles')->insertGetId([
                    'name' => $request->role_name,
                    'tenant_id' => $tenantId,
                    'created_at' => now(),
                    'updated_at' => now(),
                ]);
            }

            // 2️⃣ Create User
            $userId = DB::table('users')->insertGetId([
                'name' => $request->user_name,
                'email' => $request->user_email,
                'password' => Hash::make($request->user_password),
                'tenant_id' => $tenantId,
                'role' => 'custom',
                'created_at' => now(),
                'updated_at' => now(),
            ]);

            // 3️⃣ Link User to Role
            $roleUserId = DB::table('role_user')->insertGetId([
                'user_id' => $userId,
                'role_id' => $roleId,
                'tenant_id' => $tenantId,
                'created_at' => now(),
                'updated_at' => now(),
            ]);

            // 4️⃣ Assign Permissions Automatically
            if ($request->user_type === 'existing_role') {
                // Copy permissions from existing role
                $this->copyPermissionsFromExistingRole($roleId, $roleUserId, $tenantId);
            } else {
                // Assign permissions by groups
                $this->assignPermissionsByGroups($request->permission_groups, $roleUserId, $planId, $tenantId);
            }
            
            return [
                'user_name' => $request->user_name,
                'user_email' => $request->user_email,
                'role_type' => $request->user_type,
                'role_name' => $request->user_type === 'existing_role' ? 
                    DB::table('roles')->where('id', $roleId)->value('name') : 
                    $request->role_name
            ];
        });

        return redirect()->back()->with('success', 
            "User '{$result['user_name']}' created successfully with role '{$result['role_name']}'!");
    }
    
    /**
     * Copy permissions from an existing role to a new role_user entry
     */
    private function copyPermissionsFromExistingRole($roleId, $newRoleUserId, $tenantId)
    {
        // Find an existing role_user entry for this role to copy permissions from
        $existingRoleUserId = DB::table('role_user')
            ->where('role_id', $roleId)
            ->where('tenant_id', $tenantId)
            ->where('id', '!=', $newRoleUserId) // Exclude the newly created entry
            ->value('id');

        if ($existingRoleUserId) {
            $permissions = DB::table('permission_role')
                ->where('role_id', $existingRoleUserId)
                ->pluck('permission_id');

            foreach ($permissions as $permissionId) {
                DB::table('permission_role')->insert([
                    'role_id' => $newRoleUserId,
                    'permission_id' => $permissionId,
                    'created_at' => now(),
                    'updated_at' => now(),
                ]);
            }
        }
    }
    
    /**
     * Assign permissions to a role_user based on permission groups
     */
    private function assignPermissionsByGroups($permissionGroups, $roleUserId, $planId, $tenantId)
    {
        foreach ($permissionGroups as $groupName) {
            $permissionIds = $this->getPermissionsByGroup($groupName, $planId, $tenantId);
            
            foreach ($permissionIds as $permissionId) {
                DB::table('permission_role')->insert([
                    'role_id' => $roleUserId,
                    'permission_id' => $permissionId,
                    'created_at' => now(),
                    'updated_at' => now(),
                ]);
            }
        }
    }

    public function checkTenantLimit($tenantId)
    {
        $stats = \DB::table('tenants')
            ->join('subscription_plans', 'subscription_plans.id', '=', 'tenants.subscription_plan_id')
            ->join('users', 'users.tenant_id', '=', 'tenants.id')
            ->where('tenants.id', $tenantId)
            ->selectRaw('COUNT(users.id) as tenant_max, subscription_plans.max_users')
            ->groupBy('subscription_plans.max_users')
            ->first();

        if ($stats && $stats->tenant_max >= $stats->max_users) {
            // Show custom page
            return response()->view('errors.user_limit', [
                'tenant_max' => $stats->tenant_max,
                'max_users'  => $stats->max_users
            ]);
        }

        return null; // limit not reached
    }

    public function upgrade()
    {
        return view('subscriptions.upgrade');
    }
}
