<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\User;
use App\Models\ActivityLog;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Auth;
use App\Models\Permission;
use App\Models\Role;
use App\Models\VisitType;
use DB;

class UserController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
        $this->middleware('role:administrateur')->except(['profile', 'updateProfile']);
    }

    public function index()
    {
        $users = User::orderBy('created_at', 'desc')->paginate(10);
        return view('users.index', compact('users'));
    }

    public function create()
    {

       // return view('users.create');
    }

    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'email' => 'required|string|email|max:255|unique:users',
            'password' => 'required|string|min:6|confirmed',
            'role' => 'required|in:admin,assistant',
            'permissions' => 'array',
        ], [
            'name.required' => __('forms.field_required'),
            'email.required' => __('forms.field_required'),
            'email.email' => __('forms.invalid_email'),
            'email.unique' => __('validation.email_unique'),
            'password.required' => __('forms.field_required'),
            'password.min' => __('forms.password_min_length'),
            'password.confirmed' => __('forms.passwords_not_match'),
            'role.required' => __('forms.field_required'),
        ]);

        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        $user = User::create([
            'name' => $request->name,
            'email' => $request->email,
            'password' => Hash::make($request->password),
            'role' => $request->role,
            'permissions' => $request->permissions ?? [],
            'is_active' => true,
        ]);

        // Log the activity
        ActivityLog::create([
            'user_id' => Auth::id(),
            'action' => 'create_user',
            'description' => __('activity_log.user_created', ['name' => $user->name]),
            'ip_address' => $request->ip(),
            'user_agent' => $request->userAgent(),
        ]);



        return redirect()->route('users.index')->with('success', __('users.user_created'));
    }

    public function show(User $user)
    {
        return view('users.show', compact('user'));
    }

    public function edit(User $user)
    {

        $user->load('permissions');

        $tenantId = auth()->user()->tenant_id;
        $roles = Role::where('tenant_id', $tenantId)->get();

        return view('users.edit', compact('user', 'roles'));
    }

    public function update(Request $request, User $user)
    {


        DB::beginTransaction();

        try {
            $role_name = Role::where('id', $request->role)->value('name');
            $updateData = [
                'name' => $request->name,
                'email' => $request->email,
                'role' =>  $role_name,
                'is_active' => $request->has('is_active'),
            ];

            if ($request->filled('password')) {
                $updateData['password'] = Hash::make($request->password);
            }

            $user->update($updateData);

            // تحديث الصلاحيات
            $roleUser = DB::table('role_user')
                ->where('user_id', $user->id)
                ->first();


            if ($roleUser) {

                if ($roleUser->role_id != $request->role) {


                // حذف الصلاحيات القديمة
                DB::table('permission_role')
                    ->where('role_id', $roleUser->id)
                    ->delete();


                    $roleUserid = DB::table('role_user')
                ->where('role_id', $request->role)
                ->first();

                $rolePermissions = DB::table('permission_role')
                ->where('role_id', $roleUserid->id)
                ->get();


                // إضافة الصلاحيات الجديدة
                $permissionsData = [];
                foreach ($rolePermissions as $perm) {
                    $permissionsData[] = [  // لاحظ استخدام [] لإضافة عنصر جديد للمصفوفة
                        'role_id' => $roleUser->id,
                        'permission_id' => $perm->permission_id,
                        'created_at' => now(),
                        'updated_at' => now()
                    ];
                }

                DB::table('permission_role')->insert($permissionsData);

                DB::table('role_user')
                ->where('user_id', $user->id)
                ->update(['role_id' => $request->role]);

            }
            }

            ActivityLog::create([
                'user_id' => Auth::id(),
                'action' => 'update_user',
                'description' => __('activity_log.user_updated', ['name' => $user->name]),
                'ip_address' => $request->ip(),
                'user_agent' => $request->userAgent(),
            ]);

            DB::commit();

            return redirect()->route('users.index')->with('success', __('users.user_updated'));
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', __('users.update_error') . ': ' . $e->getMessage());
        }
    }

    // دالة مساعدة للحصول على الصلاحيات بناءً على الدور
    private function getPermissionsForRole($role)
    {
        // يمكنك استبدال هذا بمصدر بياناتك الفعلي (قاعدة بيانات، ملف config، إلخ)
        $rolePermissions = [
            'admin' => [1, 2, 3, 4], // مثال: صلاحيات المدير
            'assistant' => [2, 3]    // مثال: صلاحيات المساعد
        ];

        return $rolePermissions[$role] ?? [];
    }

    public function destroy(User $user)
    {
        if ($user->id === Auth::id()) {
            return back()->with('error', __('users.cannot_delete_self'));
        }

        $userName = $user->name;
        $user->delete();

        // Log the activity
        ActivityLog::create([
            'user_id' => Auth::id(),
            'action' => 'delete_user',
            'description' => __('activity_log.user_deleted', ['name' => $userName]),
            'ip_address' => request()->ip(),
            'user_agent' => request()->userAgent(),
        ]);

        return redirect()->route('users.index')->with('success', __('users.user_deleted'));
    }

    public function profile()
    {
        $user = Auth::user();
        return view('users.profile', compact('user'));
    }

    public function updateProfile(Request $request)
    {
        $user = Auth::user();

        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'email' => 'required|string|email|max:255|unique:users,email,' . $user->id,
            'current_password' => 'required_with:password',
            'password' => 'nullable|string|min:6|confirmed',
        ]);

        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        if ($request->filled('password')) {
            if (!Hash::check($request->current_password, $user->password)) {
                return back()->withErrors(['current_password' => __('users.current_password_incorrect')]);
            }
        }

        $updateData = [
            'name' => $request->name,
            'email' => $request->email,
        ];

        if ($request->filled('password')) {
            $updateData['password'] = Hash::make($request->password);
        }

        $user->update($updateData);

        return back()->with('success', __('users.profile_updated'));
    }
}
