<!DOCTYPE html>
<html lang="{{ app()->getLocale() }}" dir="{{ app()->getLocale() == 'ar' ? 'rtl' : 'ltr' }}">
<head>
    <meta charset="utf-8">
    <meta name="viewport" content="width=device-width, initial-scale=1">
    <meta name="csrf-token" content="{{ csrf_token() }}">
    
    <title>{{ __('auth.qr_login_title') }}</title>
    
    <!-- Bootstrap CSS -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.rtl.min.css" rel="stylesheet">
    <!-- Font Awesome -->
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css" rel="stylesheet">
    <!-- Google Fonts Arabic -->
    <link href="https://fonts.googleapis.com/css2?family=Cairo:wght@300;400;600;700&display=swap" rel="stylesheet">
    
    <style>
        body {
            font-family: 'Cairo', sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
            padding: 20px;
        }
        
        .qr-container {
            background: white;
            border-radius: 20px;
            box-shadow: 0 15px 35px rgba(0, 0, 0, 0.1);
            overflow: hidden;
            max-width: 400px;
            width: 100%;
        }
        
        .qr-header {
            background: linear-gradient(135deg, #667eea, #764ba2);
            color: white;
            text-align: center;
            padding: 30px 20px;
        }
        
        .qr-header i {
            font-size: 3rem;
            margin-bottom: 15px;
        }
        
        .qr-form {
            padding: 30px;
        }
        
        .form-floating {
            margin-bottom: 20px;
        }
        
        .form-control {
            border-radius: 15px;
            border: 2px solid #e9ecef;
            padding: 12px 20px;
            transition: all 0.3s ease;
        }
        
        .form-control:focus {
            border-color: #667eea;
            box-shadow: 0 0 0 0.2rem rgba(102, 126, 234, 0.25);
        }
        
        .btn-qr-login {
            background: linear-gradient(135deg, #667eea, #764ba2);
            border: none;
            border-radius: 15px;
            padding: 15px;
            font-size: 1.1rem;
            font-weight: 600;
            width: 100%;
            color: white;
            transition: all 0.3s ease;
        }
        
        .btn-qr-login:hover:not(:disabled) {
            background: linear-gradient(135deg, #5a6fd8, #6a4190);
            transform: translateY(-2px);
            box-shadow: 0 8px 25px rgba(102, 126, 234, 0.3);
        }
        
        .btn-qr-login:disabled {
            opacity: 0.6;
            cursor: not-allowed;
        }
        
        .alert {
            border-radius: 15px;
            border: none;
            margin-bottom: 20px;
        }
        
        .spinner-border {
            width: 1rem;
            height: 1rem;
        }
        
        .expires-info {
            background-color: #f8f9fa;
            border-radius: 10px;
            padding: 15px;
            text-align: center;
            margin-bottom: 20px;
        }
        
        .expires-info small {
            color: #6c757d;
        }
    </style>
</head>
<body>
    <div class="qr-container">
        <div class="qr-header">
            <i class="fas fa-qrcode"></i>
            <h3>{{ __('QR Code Login') }}</h3>
            <p>{{ __('Enter your credentials to complete login') }}</p>
        </div>
        
        <div class="qr-form">
            <div class="expires-info">
                <i class="fas fa-clock text-warning"></i>
                <small>{{ __('This QR code expires in') }} <span id="countdown"></span></small>
            </div>
            
            <div id="alert-container"></div>
            
            <form id="qr-auth-form">
                @csrf
                <div class="form-floating">
                    <input type="email" class="form-control" id="email" name="email" 
                           placeholder="{{ __('auth.email') }}" required>
                    <label for="email">
                        <i class="fas fa-envelope me-2"></i>{{ __('auth.email') }}
                    </label>
                </div>
                
                <div class="form-floating">
                    <input type="password" class="form-control" id="password" name="password" 
                           placeholder="{{ __('auth.password') }}" required>
                    <label for="password">
                        <i class="fas fa-lock me-2"></i>{{ __('auth.password') }}
                    </label>
                </div>
                
                <button type="submit" id="auth-btn" class="btn btn-qr-login">
                    <span class="btn-text">
                        <i class="fas fa-sign-in-alt me-2"></i>
                        {{ __('Complete Login') }}
                    </span>
                    <span class="btn-spinner d-none">
                        <span class="spinner-border me-2" role="status"></span>
                        {{ __('Processing...') }}
                    </span>
                </button>
            </form>
            
            <div class="mt-3 text-center">
                <small class="text-muted">
                    <i class="fas fa-shield-alt me-1"></i>
                    {{ __('Secure QR Authentication') }}
                </small>
            </div>
        </div>
    </div>
    
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        const token = '{{ $qrToken->token }}';
        const expiresAt = new Date('{{ $qrToken->expires_at->toISOString() }}');
        let countdownInterval;
        
        // Countdown timer
        function updateCountdown() {
            const now = new Date();
            const timeLeft = expiresAt - now;
            
            if (timeLeft <= 0) {
                showAlert('QR code has expired. Please generate a new one.', 'danger');
                document.getElementById('qr-auth-form').style.display = 'none';
                clearInterval(countdownInterval);
                return;
            }
            
            const minutes = Math.floor(timeLeft / (1000 * 60));
            const seconds = Math.floor((timeLeft % (1000 * 60)) / 1000);
            
            document.getElementById('countdown').textContent = 
                `${minutes}:${seconds.toString().padStart(2, '0')}`;
        }
        
        countdownInterval = setInterval(updateCountdown, 1000);
        updateCountdown();
        
        // Handle form submission
        document.getElementById('qr-auth-form').addEventListener('submit', async function(e) {
            e.preventDefault();
            
            const btn = document.getElementById('auth-btn');
            const btnText = btn.querySelector('.btn-text');
            const btnSpinner = btn.querySelector('.btn-spinner');
            
            // Show loading state
            btn.disabled = true;
            btnText.classList.add('d-none');
            btnSpinner.classList.remove('d-none');
            
            const formData = new FormData(this);
            
            try {
                const response = await fetch(`/qr-login/${token}/authenticate`, {
                    method: 'POST',
                    body: formData,
                    headers: {
                        'X-Requested-With': 'XMLHttpRequest',
                        'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content
                    }
                });
                
                const data = await response.json();
                
                if (data.success) {
                    showAlert(`${data.message}<br><strong>${data.user_name}</strong>`, 'success');
                    document.getElementById('qr-auth-form').style.display = 'none';
                    
                    // Show success message for a few seconds
                    setTimeout(() => {
                        window.close(); // Try to close the tab/window
                    }, 3000);
                } else {
                    showAlert(data.message, 'danger');
                }
            } catch (error) {
                showAlert('An error occurred. Please try again.', 'danger');
            } finally {
                // Reset button state
                btn.disabled = false;
                btnText.classList.remove('d-none');
                btnSpinner.classList.add('d-none');
            }
        });
        
        function showAlert(message, type) {
            const alertContainer = document.getElementById('alert-container');
            alertContainer.innerHTML = `
                <div class="alert alert-${type}" role="alert">
                    <i class="fas fa-${type === 'success' ? 'check-circle' : 'exclamation-triangle'} me-2"></i>
                    ${message}
                </div>
            `;
        }
    </script>
</body>
</html>