<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>{{ __('display.pairing_title', ['name' => $display->name]) }}</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <style>
        body {
            font-family: 'Inter', sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            margin: 0;
            padding: 0;
            overflow: hidden;
        }
        
        .pairing-container {
            display: flex;
            align-items: center;
            justify-content: center;
            min-height: 100vh;
            padding: 2rem;
        }
        
        .pairing-card {
            background: white;
            border-radius: 24px;
            box-shadow: 0 20px 60px rgba(0, 0, 0, 0.1);
            max-width: 800px;
            width: 100%;
            text-align: center;
            padding: 3rem;
            position: relative;
            overflow: hidden;
        }
        
        .pairing-card::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            width: 100%;
            height: 4px;
            background: linear-gradient(90deg, #667eea, #764ba2);
        }
        
        .status-indicator {
            display: inline-flex;
            align-items: center;
            gap: 8px;
            padding: 8px 16px;
            background: #f8f9fa;
            border-radius: 20px;
            font-size: 0.9rem;
            margin-bottom: 2rem;
        }
        
        .status-dot {
            width: 8px;
            height: 8px;
            border-radius: 50%;
            background: #ffc107;
            animation: pulse 2s infinite;
        }
        
        .status-dot.waiting { background: #ffc107; }
        .status-dot.in-progress { background: #17a2b8; }
        .status-dot.completed { background: #28a745; }
        
        @keyframes pulse {
            0%, 100% { opacity: 1; }
            50% { opacity: 0.5; }
        }
        
        .display-info {
            margin-bottom: 2rem;
        }
        
        .display-name {
            font-size: 2rem;
            font-weight: 600;
            color: #2d3748;
            margin-bottom: 0.5rem;
        }
        
        .display-subtitle {
            font-size: 1.1rem;
            color: #718096;
            margin-bottom: 3rem;
        }
        
        .qr-section {
            margin: 3rem 0;
        }
        
        .qr-container {
            display: inline-block;
            padding: 1.5rem;
            background: #f8f9fa;
            border-radius: 16px;
            margin-bottom: 1.5rem;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.05);
        }
        
        .instructions {
            max-width: 600px;
            margin: 0 auto;
        }
        
        .instruction-step {
            display: flex;
            align-items: center;
            text-align: left;
            margin-bottom: 1rem;
            padding: 1rem;
            background: #f8f9fa;
            border-radius: 12px;
        }
        
        .step-number {
            width: 32px;
            height: 32px;
            background: #667eea;
            color: white;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-weight: 600;
            margin-right: 1rem;
            flex-shrink: 0;
        }
        
        .step-text {
            font-size: 1rem;
            color: #4a5568;
        }
        
        .short-url {
            background: #e6fffa;
            border: 2px dashed #38b2ac;
            border-radius: 8px;
            padding: 1rem;
            font-family: monospace;
            font-size: 1.2rem;
            font-weight: 600;
            color: #2d3748;
            margin: 2rem 0;
        }
        
        .footer-info {
            margin-top: 3rem;
            padding-top: 2rem;
            border-top: 1px solid #e2e8f0;
            color: #718096;
            font-size: 0.9rem;
        }
        
        .loading-spinner {
            display: none;
            margin: 2rem auto;
        }
        
        .success-animation {
            display: none;
            margin: 2rem 0;
        }
        
        .success-icon {
            width: 80px;
            height: 80px;
            background: #28a745;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            margin: 0 auto 1rem;
            animation: bounce 0.6s ease-out;
        }
        
        @keyframes bounce {
            0%, 20%, 50%, 80%, 100% { transform: translateY(0); }
            40% { transform: translateY(-10px); }
            60% { transform: translateY(-5px); }
        }
        
        .reset-button {
            position: absolute;
            top: 1rem;
            right: 1rem;
            background: none;
            border: none;
            color: #718096;
            font-size: 1.2rem;
            padding: 0.5rem;
            border-radius: 8px;
            cursor: pointer;
            transition: all 0.2s;
        }
        
        .reset-button:hover {
            background: #f7fafc;
            color: #4a5568;
        }
    </style>
</head>
<body>
    <div class="pairing-container">
        <div class="pairing-card">
            <button class="reset-button" onclick="resetPairing()" title="Start Over">
                <i class="fas fa-refresh"></i>
            </button>
            
            <div class="status-indicator">
                <div class="status-dot {{ $display->pairing_status ?? 'waiting' }}" id="statusDot"></div>
                <span id="statusText">
                    @if(($display->pairing_status ?? 'waiting') === 'waiting')
                        Waiting for mobile scan
                    @elseif($display->pairing_status === 'in_progress')
                        Pairing in progress...
                    @else
                        Pairing completed!
                    @endif
                </span>
            </div>
            
            <div class="display-info">
                <h1 class="display-name">{{ $display->name ?? 'Display Device' }}</h1>
                <p class="display-subtitle">Pair this display with your account</p>
            </div>
            
            <div id="pairingContent">
                <div class="qr-section">
                    <div class="qr-container">
                        {!! $qrCode !!}
                    </div>
                    <p style="color: #718096; margin-bottom: 2rem;">
                        Scan this QR code with your mobile device
                    </p>
                </div>
                
                <div class="instructions">
                    <div class="instruction-step">
                        <div class="step-number">1</div>
                        <div class="step-text">Type the short link on this TV/display screen</div>
                    </div>
                    <div class="instruction-step">
                        <div class="step-number">2</div>
                        <div class="step-text">Scan the QR code above with your mobile device</div>
                    </div>
                    <div class="instruction-step">
                        <div class="step-number">3</div>
                        <div class="step-text">Enter your email and password on mobile</div>
                    </div>
                    <div class="instruction-step">
                        <div class="step-number">4</div>
                        <div class="step-text">TV will automatically redirect to display view</div>
                    </div>
                </div>
            </div>
            
            <div class="loading-spinner" id="loadingSpinner">
                <div class="spinner-border text-primary" role="status">
                    <span class="visually-hidden">Loading...</span>
                </div>
                <p class="mt-2">Completing pairing...</p>
            </div>
            
            <div class="success-animation" id="successAnimation">
                <div class="success-icon">
                    <i class="fas fa-check text-white" style="font-size: 2rem;"></i>
                </div>
                <h3 class="text-success">Pairing Successful!</h3>
                <p class="text-muted">Redirecting to display view...</p>
            </div>
            
            <div class="footer-info">
                <p>This pairing session expires in 10 minutes</p>
                <p><small>Display ID: {{ substr($display->device_token, 0, 12) }}...</small></p>
            </div>
        </div>
    </div>
    
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        let pairingStatus = '{{ $display->pairing_status ?? "waiting" }}';
        let pollInterval;
        
        function startPolling() {
            pollInterval = setInterval(checkPairingStatus, 2000); // Check every 2 seconds
        }
        
        function stopPolling() {
            if (pollInterval) {
                clearInterval(pollInterval);
            }
        }
        
        async function checkPairingStatus() {
            try {
                const response = await fetch('{{ route('display.pairing.status', $display->device_token) }}');
                const data = await response.json();
                
                if (data.success) {
                    updatePairingStatus(data.status);
                    
                    if (data.should_redirect) {
                        // Pairing completed successfully
                        showSuccess();
                        setTimeout(() => {
                            window.location.href = '{{ route('display.index', $display->device_token) }}';
                        }, 3000);
                    }
                }
            } catch (error) {
                console.error('Error checking pairing status:', error);
            }
        }
        
        function updatePairingStatus(status) {
            if (status === pairingStatus) return;
            
            pairingStatus = status;
            const statusDot = document.getElementById('statusDot');
            const statusText = document.getElementById('statusText');
            
            // Update status indicator
            statusDot.className = 'status-dot ' + status;
            
            switch(status) {
                case 'waiting':
                    statusText.textContent = 'Waiting for mobile scan';
                    break;
                case 'in_progress':
                    statusText.textContent = 'Pairing in progress...';
                    showLoading();
                    break;
                case 'completed':
                    statusText.textContent = 'Pairing completed!';
                    break;
            }
        }
        
        function showLoading() {
            document.getElementById('pairingContent').style.display = 'none';
            document.getElementById('loadingSpinner').style.display = 'block';
        }
        
        function showSuccess() {
            document.getElementById('loadingSpinner').style.display = 'none';
            document.getElementById('successAnimation').style.display = 'block';
            stopPolling();
        }
        
        function resetPairing() {
            if (confirm('Are you sure you want to start the pairing process over?')) {
                fetch('{{ route('display.pairing.reset', $display->device_token) }}', {
                    method: 'POST',
                    headers: {
                        'X-CSRF-TOKEN': '{{ csrf_token() }}',
                        'Content-Type': 'application/json'
                    }
                }).then(() => {
                    window.location.reload();
                });
            }
        }
        
        // Start polling when page loads
        document.addEventListener('DOMContentLoaded', function() {
            startPolling();
        });
        
        // Clean up on page unload
        window.addEventListener('beforeunload', function() {
            stopPolling();
        });
    </script>
</body>
</html>
