@extends('layouts.app')

@section('content')
<div class="container-fluid">
    <div class="d-flex justify-content-between align-items-center mb-4">
        <div>
            <h1 class="h3 mb-0">{{ __('Join Requests') }}</h1>
            <p class="text-muted small mb-0">{{ __('Manage pending patient requests to join the waiting room') }}</p>
        </div>
        <div>
            <span class="badge bg-warning fs-6" id="pendingCount">{{ $requests->total() }}</span>
            <span class="text-muted small">{{ __('Pending Requests') }}</span>
        </div>
    </div>

    @if($requests->count() > 0)
        <div class="card">
            <div class="card-header">
                <h5 class="card-title mb-0">{{ __('Pending Join Requests') }}</h5>
            </div>
            <div class="card-body p-0">
                <div class="table-responsive">
                    <table class="table table-hover mb-0">
                        <thead class="table-light">
                            <tr>
                                <th>{{ __('Patient') }}</th>
                                <th>{{ __('Contact') }}</th>
                                <th>{{ __('Location') }}</th>
                                <th>{{ __('Requested At') }}</th>
                                <th class="text-center">{{ __('Actions') }}</th>
                            </tr>
                        </thead>
                        <tbody>
                            @foreach($requests as $request)
                            <tr id="request-row-{{ $request->id }}">
                                <td>
                                    <div>
                                        <strong>{{ $request->patient->name }}</strong>
                                    </div>
                                    @if($request->patient->email)
                                        <small class="text-muted">{{ $request->patient->email }}</small>
                                    @endif
                                </td>
                                <td>
                                    @if($request->patient->phone)
                                        <span class="text-muted">
                                            <i class="fas fa-phone fa-sm me-1"></i>{{ $request->patient->phone }}
                                        </span>
                                    @else
                                        <span class="text-muted">{{ __('No phone') }}</span>
                                    @endif
                                </td>
                                <td>
                                    <div>
                                        <strong>{{ $request->tenant->name }}</strong>
                                    </div>
                                    @if($request->tenant->address)
                                        <small class="text-muted">{{ $request->tenant->address }}</small>
                                    @endif
                                </td>
                                <td>
                                    <div>{{ $request->requested_at->format('H:i') }}</div>
                                    <small class="text-muted">{{ $request->requested_at->format('d/m/Y') }}</small>
                                    <div class="small text-info">{{ $request->requested_at->diffForHumans() }}</div>
                                </td>
                                <td class="text-center">
                                    <div class="btn-group" role="group">
                                        <button type="button" class="btn btn-success btn-sm" 
                                                onclick="approveRequest('{{ $request->id }}')" 
                                                id="approve-btn-{{ $request->id }}">
                                            <i class="fas fa-check me-1"></i>{{ __('Approve') }}
                                        </button>
                                        <button type="button" class="btn btn-danger btn-sm" 
                                                onclick="rejectRequest('{{ $request->id }}')" 
                                                id="reject-btn-{{ $request->id }}">
                                            <i class="fas fa-times me-1"></i>{{ __('Reject') }}
                                        </button>
                                    </div>
                                </td>
                            </tr>
                            @endforeach
                        </tbody>
                    </table>
                </div>
            </div>
            
            @if($requests->hasPages())
                <div class="card-footer">
                    {{ $requests->links() }}
                </div>
            @endif
        </div>
    @else
        <div class="card">
            <div class="card-body text-center py-5">
                <i class="fas fa-inbox fa-3x text-muted mb-3"></i>
                <h5 class="text-muted">{{ __('No Pending Requests') }}</h5>
                <p class="text-muted">{{ __('All join requests have been processed or there are no new requests.') }}</p>
            </div>
        </div>
    @endif
</div>

<!-- Approval Modal -->
<div class="modal fade" id="approvalModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="approvalModalTitle">{{ __('Approve Join Request') }}</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <div class="mb-3">
                    <label for="visitTypeSelect" class="form-label fw-semibold">
                        <i class="fas fa-stethoscope text-primary me-1"></i>
                        {{ __('Visit Type') }}
                    </label>
                    <select class="form-select form-select-lg" id="visitTypeSelect">
                        <option value="">{{ __('Select visit type (optional)') }}</option>
                        <!-- Visit types will be populated dynamically -->
                    </select>
                    <div class="form-text">{{ __('Choose the appropriate visit type for this patient') }}</div>
                </div>
                <div class="mb-3">
                    <label for="approvalNotes" class="form-label">{{ __('Notes (Optional)') }}</label>
                    <textarea class="form-control" id="approvalNotes" rows="3" 
                              placeholder="{{ __('Add any notes about this approval...') }}"></textarea>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">{{ __('Cancel') }}</button>
                <button type="button" class="btn btn-success" onclick="confirmApproval()">{{ __('Approve Request') }}</button>
            </div>
        </div>
    </div>
</div>

<!-- Rejection Modal -->
<div class="modal fade" id="rejectionModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">{{ __('Reject Join Request') }}</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <div class="mb-3">
                    <label for="rejectionNotes" class="form-label">{{ __('Reason for Rejection') }}</label>
                    <textarea class="form-control" id="rejectionNotes" rows="3" 
                              placeholder="{{ __('Please provide a reason for rejecting this request...') }}" required></textarea>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">{{ __('Cancel') }}</button>
                <button type="button" class="btn btn-danger" onclick="confirmRejection()">{{ __('Reject Request') }}</button>
            </div>
        </div>
    </div>
</div>

@endsection

@push('scripts')
<script>
let currentRequestId = null;
let currentRequestData = null;

// Store request data for easy access
const requestsData = {
    @foreach($requests as $request)
    {{ $request->id }}: {
        id: {{ $request->id }},
        tenant_name: '{{ $request->tenant->name }}',
        visit_types: [
            @foreach($request->tenant->visitTypes as $visitType)
            {
                name: '{{ $visitType->name }}',
                duration: {{ $visitType->estimated_duration ?? 'null' }}
            },
            @endforeach
        ]
    },
    @endforeach
};

function approveRequest(requestId) {
    currentRequestId = requestId;
    currentRequestData = requestsData[requestId];
    
    // Clear previous values
    document.getElementById('approvalNotes').value = '';
    document.getElementById('visitTypeSelect').innerHTML = '<option value="">{{ __('Select visit type (optional)') }}</option>';
    
    // Update modal title with tenant name
    const modalTitle = document.getElementById('approvalModalTitle');
    if (currentRequestData) {
        modalTitle.textContent = '{{ __('Approve Join Request') }} - ' + currentRequestData.tenant_name;
    }
    
    // Populate visit types for this tenant
    if (currentRequestData && currentRequestData.visit_types) {
        currentRequestData.visit_types.forEach(function(visitType) {
            const option = document.createElement('option');
            option.value = visitType.name;
            option.textContent = visitType.name + (visitType.duration ? ` (${visitType.duration} min)` : '');
            document.getElementById('visitTypeSelect').appendChild(option);
        });
    }
    
    new bootstrap.Modal(document.getElementById('approvalModal')).show();
}

function rejectRequest(requestId) {
    currentRequestId = requestId;
    document.getElementById('rejectionNotes').value = '';
    new bootstrap.Modal(document.getElementById('rejectionModal')).show();
}

function confirmApproval() {
    if (!currentRequestId) return;
    
    const notes = document.getElementById('approvalNotes').value;
    const visitType = document.getElementById('visitTypeSelect').value;
    const approveBtn = document.getElementById(`approve-btn-${currentRequestId}`);
    
    // Disable buttons
    approveBtn.disabled = true;
    approveBtn.innerHTML = '<i class="fas fa-spinner fa-spin me-1"></i>{{ __("Processing...") }}';
    
    fetch(`/join-requests/${currentRequestId}/approve`, {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content
        },
        body: JSON.stringify({ 
            notes: notes,
            visit_type: visitType 
        })
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            // Remove the row
            document.getElementById(`request-row-${currentRequestId}`).remove();
            
            // Update pending count
            const countEl = document.getElementById('pendingCount');
            countEl.textContent = parseInt(countEl.textContent) - 1;
            
            // Show success message
            showAlert('success', data.message || '{{ __("Request approved successfully!") }}');
            
            // Close modal
            bootstrap.Modal.getInstance(document.getElementById('approvalModal')).hide();
            
            // Check if no more requests
            checkEmptyState();
        } else {
            showAlert('danger', data.message || '{{ __("Failed to approve request") }}');
            approveBtn.disabled = false;
            approveBtn.innerHTML = '<i class="fas fa-check me-1"></i>{{ __("Approve") }}';
        }
    })
    .catch(error => {
        console.error('Error:', error);
        showAlert('danger', '{{ __("An error occurred") }}');
        approveBtn.disabled = false;
        approveBtn.innerHTML = '<i class="fas fa-check me-1"></i>{{ __("Approve") }}';
    });
}

function confirmRejection() {
    if (!currentRequestId) return;
    
    const notes = document.getElementById('rejectionNotes').value.trim();
    if (!notes) {
        showAlert('warning', '{{ __("Please provide a reason for rejection") }}');
        return;
    }
    
    const rejectBtn = document.getElementById(`reject-btn-${currentRequestId}`);
    
    // Disable buttons
    rejectBtn.disabled = true;
    rejectBtn.innerHTML = '<i class="fas fa-spinner fa-spin me-1"></i>{{ __("Processing...") }}';
    
    fetch(`/join-requests/${currentRequestId}/reject`, {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content
        },
        body: JSON.stringify({ notes: notes })
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            // Remove the row
            document.getElementById(`request-row-${currentRequestId}`).remove();
            
            // Update pending count
            const countEl = document.getElementById('pendingCount');
            countEl.textContent = parseInt(countEl.textContent) - 1;
            
            // Show success message
            showAlert('success', data.message || '{{ __("Request rejected successfully!") }}');
            
            // Close modal
            bootstrap.Modal.getInstance(document.getElementById('rejectionModal')).hide();
            
            // Check if no more requests
            checkEmptyState();
        } else {
            showAlert('danger', data.message || '{{ __("Failed to reject request") }}');
            rejectBtn.disabled = false;
            rejectBtn.innerHTML = '<i class="fas fa-times me-1"></i>{{ __("Reject") }}';
        }
    })
    .catch(error => {
        console.error('Error:', error);
        showAlert('danger', '{{ __("An error occurred") }}');
        rejectBtn.disabled = false;
        rejectBtn.innerHTML = '<i class="fas fa-times me-1"></i>{{ __("Reject") }}';
    });
}

function showAlert(type, message) {
    const alertDiv = document.createElement('div');
    alertDiv.className = `alert alert-${type} alert-dismissible fade show`;
    alertDiv.innerHTML = `
        ${message}
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    `;
    
    const container = document.querySelector('.container-fluid');
    container.insertBefore(alertDiv, container.firstChild);
    
    setTimeout(() => {
        alertDiv.remove();
    }, 5000);
}

function checkEmptyState() {
    const tableBody = document.querySelector('tbody');
    if (tableBody && tableBody.children.length === 0) {
        location.reload(); // Reload to show empty state
    }
}

// Auto-refresh pending count every 30 seconds
setInterval(() => {
    fetch('/join-requests/api/pending-count')
        .then(response => response.json())
        .then(data => {
            if (data.pending_count !== undefined) {
                document.getElementById('pendingCount').textContent = data.pending_count;
            }
        })
        .catch(error => console.error('Error updating pending count:', error));
}, 30000);
</script>
@endpush